/*
 * getpath.c
 * File revision 4
 * Process default paths a bit more intelligently.
 * (c) 2000,2001 Jacob Lundberg, jacob@chaos2.org
 */


/*
 * 1999.10.24	original framework
 * 2000.03.15	Complete rewrite
 *		reduced complexity (needs reduced more)
 * 2000.03.24	split parse() into subfunctions (finally)
 * 2000.07.24	moved includes here
 *		updated coding style
 *		made first requests take ordering priority
 * 2001.08.30	cleaned things up some more
 */


#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>                                                                                                              
#include "getpath.h"


int main(int argc, char **argv) {
/*
 * main()
 * Fill the paths into a stack, output the stack.
 */

   int counter;
   char *homepath;
   pathlist *head = NULL;
   struct passwd *pwentry = NULL;

   pwentry = getpwuid(geteuid());
   homepath = (char *)malloc(strlen(pwentry->pw_dir) + strlen(HOMEFILE) + 2);
   sprintf(homepath, "%s/%s", pwentry->pw_dir, HOMEFILE);

   if(!parse(&head, ETCFILE) && !parse(&head, homepath)) {
      for(counter = 1; counter < argc; counter++)
         if(!strcasecmp(argv[counter], "--help") || !strcasecmp(argv[counter], "-h"))
            help();
         else if(parse(&head, argv[counter]))
            break;
   }

   /* output some sort of path */
   if(head == NULL || head->item == NULL)
      defaults();
   else
      output(head);

   /* mem will clean itself; freeing the list would waste time */
   free(homepath);
   return(0);

}


int badpath(char *dirname) {
/*
 * badpath()
 * Test whether a dirname exists and is available to the current user.
 */

   struct stat statdir;

   if(stat(dirname, &statdir))
      return(1);
   if(!S_ISDIR(statdir.st_mode))
      return(2);

   return(0);

}


void output(pathlist *head) {
/*
 * output()
 * Output the path list.
 */

   printf(head->item);
   head = head->next;
   while(head != NULL) {
      printf(":%s", head->item);
      head = head->next;
   }

}


pathlist *list_scan(pathlist *top, char *data) {
/*
 * list_scan()
 * Scan a path slist for an item.
 */

   while(top != NULL) {
      if(!strcmp(data, top->item))
         return(top);
      else
         top = top->next;
   }

   /* if we make it down here, it's not on the list */
   return(NULL);

}


void list_insert(pathlist **top, char *data) {
/*
 * list_insert()
 * Insert an item into a path slist.
 * If it's already there, it is left in place.
 */

   pathlist *temp;

   if(list_scan(*top, data) != NULL)
      return;
   temp = (pathlist *)malloc(sizeof(pathlist));
   temp->item = (char *)malloc(strlen(data) + 1);
   strcpy(temp->item, data);
   temp->next = *top;
   *top = temp;

}


void list_append(pathlist **top, char *data) {
/*
 * list_append()
 * Append an item onto a path slist.
 * If it's already there, it is left in place.
 */

   pathlist *temp;

   if(list_scan(*top, data) != NULL)
      return;
   if(*top == NULL) {
      *top = temp = (pathlist *)malloc(sizeof(pathlist));
   } else {
      temp = *top;
      while(temp->next != NULL) temp = temp->next;
      temp->next = (pathlist *)malloc(sizeof(pathlist));
      temp = temp->next;
   }
   temp->item = (char *)malloc(strlen(data) + 1);
   strcpy(temp->item, data);
   temp->next = NULL;

}


void list_delete(pathlist **top, char *data) {
/*
 * list_delete()
 * Delete an item from a path slist.
 */

   pathlist *temp, *holder;
   temp = *top;

   if(temp == NULL)
      return;
   while(temp->next != NULL) {
      if(!strcmp(data, temp->next->item)) {
         holder = temp->next->next;
         free(temp->next->item);
         free(temp->next);
         temp->next = holder;
      } else {
         temp = temp->next;
      }
   }
   if(!strcmp(data, (*top)->item)) {
      temp = *top;
      *top = (*top)->next;
      free(temp->item);
      free(temp);
   }

}


void list_clear(pathlist **top) {
/*
 * list_clear()
 * Clear an entire path slist.
 */

   pathlist *temp;

   while(*top != NULL) {
      temp = *top;
      *top = (*top)->next;
      free(temp->item);
      free(temp);
   }

}


int checkuser(char *data, int isuser) {
/*
 * checkuser()
 * Determine if a string corresponds to a change of user.
 */

   char *tmpusr, *beg, *end;
   struct passwd *pwentry = NULL;

   beg = strchr(data, '[');
   end = strchr(data, ']');
   if(beg == NULL || end == NULL || beg > end)
      return(isuser);
   tmpusr = (char *)malloc(end - beg);
   strncpy(tmpusr, beg + 1, end - beg - 1);
   tmpusr[end - beg - 1] = '\0';
   pwentry = getpwnam(tmpusr);
   if(!strcasecmp(tmpusr, "default") || (pwentry != NULL && geteuid() == pwentry->pw_uid))
      isuser = -1;
   else
      isuser = 0;

   free(tmpusr);
   return(isuser);

}


int parse(pathlist **top, char *filename) {
/*
 * parse()
 * Parse and process a conf file
 * Memory allocation failures here will be silent.
 */

   int isuser = -1;
   FILE *file = NULL;
   struct stat statfile;
   char *tmpstr = NULL, *wrkstr = NULL;

   /* verify that we can read the file and that it's not empty */
   if(stat(filename, &statfile) || !statfile.st_size || (file = fopen(filename, "r")) == NULL)
      return(0);

   tmpstr = (char *)malloc(statfile.st_size);
   while(tmpstr != NULL && fgets(tmpstr, statfile.st_size, file) != NULL) {
      wrkstr = strtok(tmpstr, SEPARATORS);
      if(wrkstr == NULL || strlen(wrkstr) == 0) {
         /* whitespace line (special case of comment) */
      } else if(!strcasecmp(wrkstr, "insert")) {
         /* INSERT OPERATION */
         while(isuser && (wrkstr = strtok(NULL, SEPARATORS)) != NULL)
            if(!badpath(wrkstr))
               list_insert(top, wrkstr);
      } else if(!strcasecmp(wrkstr, "append")) {
         /* APPEND OPERATION */
         while(isuser && (wrkstr = strtok(NULL, SEPARATORS)) != NULL)
            if(!badpath(wrkstr))
               list_append(top, wrkstr);
      } else if(!strcasecmp(wrkstr, "delete")) {
         /* DELETE OPERATION */
         while(isuser && (wrkstr = strtok(NULL, SEPARATORS)) != NULL)
            list_delete(top, wrkstr);
      } else if(!strcasecmp(wrkstr, "clear")) {
         /* behead all our children */
         if(isuser)
            list_clear(top);
      } else if(!strcasecmp(wrkstr, "break")) {
         /* no more paths will be allowed */
         if(isuser)
            return(-2);
      } else {
         /* either a change of user or a comment */
         isuser = checkuser(wrkstr, isuser);
      }
   }
   fclose(file);
   if(tmpstr != NULL)
      free(tmpstr);

   return(0);

}


int scanlist(pathlist *head) {
/*
 * scanlist()
 * This is for debug; call it to test whether a list is viable.
 */

   int badflag = 0;

   printf("DEBUG: scanlist()\nStarting list printout...\n");
   printf("Top char sample (%c)\n", head == NULL || head->item == NULL ? '!' : head->item[0]);
   while(head != NULL) {
      printf("Item.\n");
      if(head->item == NULL) {
         ++badflag;
         break;
      }
      printf("\tThis listitem address (%p)\n", head);
      printf("\tThis listitem data (");
      printf("%s", head->item == NULL ? "!" : head->item);
      printf(")\n");
      printf("\tNext listitem address (%p)\n", head->next);
      head = head->next;
      printf("\tSuccessful next pointer assignment...\n");
   }
   printf("Ending successful list printout...\n");
   if(badflag) {
      printf("NOTE: %i bad items (data) found.\n", badflag);
      return(badflag);
   }
   fflush(stdout);
   return(0);

}


void defaults(void) {
/*
 * defaults()
 * Spit out the default path and exit.
 */

   fprintf(stderr, "getpath: no path info found; using default!\n");
   printf(DEFPATH);
   fflush(stdout);
   exit(0);

}


void help(void) {
/*
 * help()
 * Spit out the help blurb and exit.
 */

   printf("getpath v" VERSION "\n\n"
          "Usage:\texport PATH=`getpath [FILE] [...]`\n"
          "\tsetenv PATH `getpath [FILE] [...]`\n\n"
          "The string returned will be a colon-delimited path.\n"
          "The path will be taken from " ETCFILE " and then\n"
          "from ~/" HOMEFILE " and then from FILE.  If no path information\n"
          "can be found in any of these, the compile-time default\n"
          "path will be used.  See the getpath(8) man page.\n\n"
          "Report bugs to " MAINTAINER ".\n");
   fflush(stdout);
   exit(0);

}
